<?php
/**
* 2007-2018 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    PrestaShop SA <contact@prestashop.com>
*  @copyright 2007-2018 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

if (!defined('_PS_VERSION_')) {
    exit;
}

class UptainConnect extends Module
{
    protected $config_form = false;

    private $html;

    public function __construct()
    {
        $this->name = 'uptainconnect';
        $this->tab = 'advertising_marketing';
        $this->version = '1.0.4';
        $this->author = 'uptain GmbH';
        $this->need_instance = 0;
        $this->module_key = '8ac409f9d3ff0d12a5e1fb36d7f987df';

        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('uptain Connect');

        // These declarations have to be on single lines for presta to be able to translate them.
        $this->description = $this->l('We develop automated state-of-the-art solutions for regaining lost customers coupled with absolute customer orientation.');

        $this->confirmUninstall = $this->l('If you encountered a problem with the uptain service, want to suspend the usage of it or have any questions you can contact us.');

        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);

        $this->html = '';
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        return parent::install() &&
            $this->registerHook('displayBeforeBodyClosingTag');
    }

    public function uninstall()
    {
        return parent::uninstall();
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        $output = null;
        /**
         * If values have been submitted in the form, process.
         */
        if (((bool)Tools::isSubmit('submitUptainConnectModule')) == true) {
            $output .= $this->displayConfirmation($this->l('Settings updated'));
            $this->postProcess();
        }

        $this->context->smarty->assign('module_dir', $this->_path);

        $output .= $this->context->smarty->fetch(
            $this->local_path . 'views/templates/admin/configure.tpl'
        );

        return $output . $this->renderForm();
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitUptainConnectModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($this->getConfigForm()));
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        return array(
            'form' => array(
                'legend' => array(
                'title' => $this->l('Settings'),
                'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'col' => 3,
                        'type' => 'text',
                        'desc' => $this->l('Enter your personal uptain Token'),
                        'name' => 'UPTAINCONNECT_TOKEN',
                        'label' => $this->l('uptain Token'),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return array(
            'UPTAINCONNECT_TOKEN' => Configuration::get('UPTAINCONNECT_TOKEN', ''),
        );
    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();

        foreach (array_keys($form_values) as $key) {
            Configuration::updateValue($key, Tools::getValue($key));
        }
    }

    /**
    * Add the CSS & JavaScript files you want to be loaded in the BO.
    */
    public function hookBackOfficeHeader()
    {
        if (Tools::getValue('module_name') == $this->name) {
            $this->context->controller->addJS($this->_path.'views/js/back.js');
            $this->context->controller->addCSS($this->_path.'views/css/back.css');
        }
    }

    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function hookHeader()
    {
        $this->context->controller->addJS($this->_path.'/views/js/front.js');
        $this->context->controller->addCSS($this->_path.'/views/css/front.css');
    }

    private function getSCV($cart)
    {
        if (!$cart) {
            return "";
        }

        return number_format($cart->getOrderTotal(false, Cart::BOTH_WITHOUT_SHIPPING), 2, '.', '');
    }

    private function getCurrentOrder()
    {
        $cart_id = (int) (Tools::getValue('id_cart', 0));
        $order_id = Order::getIdByCartId($cart_id);
        $order = new Order((int) ($order_id));

        return $order;
    }

    private function getVouchersFromCartRules($rules)
    {
        if (!$rules) {
            return "";
        }

        $vouchers = array();

        foreach ($rules as $rule) {
            if (((float)$rule['reduction_amount']) > 0 || ((float)$rule['reduction_percent']) > 0) {
                $vouchers[] = $rule['code'];
            }
        }

        return implode(',', $vouchers);
    }

    private function getVoucherDiscountFromCartRules($rules)
    {
      if (!$rules) {
        return 0;
      }

      $discount = 0;

      foreach ($rules as $rule) {
        if ((float)$rule['value_tax_excl'] > 0) {
          $discount += (float)$rule['value_tax_excl'];
        }
      }

      return $discount;
    }

    private function getVoucher($cart)
    {
        if (!$cart) {
            return "";
        }

        $cart_rules = $cart->getCartRules();

        return $this->getVouchersFromCartRules($cart_rules);
    }

    private function getCurrency()
    {
        try {
            $debug_currency = new Currency($this->context->cart->id_currency);
            return $debug_currency->iso_code;
        } catch (Exception $e) {
            return "";
        }
    }

    private function scvFromOrder($order)
    {
        $totalWithoutDiscount = $order->getTotalProductsWithoutTaxes();

        $discount = $this->getVoucherDiscountFromCartRules($order->getCartRules());

        return $totalWithoutDiscount - $discount;
    }

    /**
     * Add the uptain snippet to the end of the body tag
     */
    public function hookDisplayBeforeBodyClosingTag($params)
    {
        $this->context->smarty->assign('uptain_module_version', $this->version);

        $cart = $this->context->cart;
        $currentOrder = $this->getCurrentOrder();

        $dataFields = array();

        if ($currentOrder && null !== $currentOrder->current_state) {
            $dataFields[] = "data-success=" . 1;
            $dataFields[] = "data-ordernumber=" . $currentOrder->reference;
            $dataFields[] = "data-scv=" . number_format($this->scvFromOrder($currentOrder), 2, '.', '');
            $dataFields[] = "debug-cartrules=\"" . $this->getVoucherDiscountFromCartRules($currentOrder->getCartRules()) . "\"";
        } else {
            $dataFields[] = "data-scv=" . $this->getSCV($cart);
            if ($this->getVoucher($cart) != "") {
                $dataFields[] = "data-usedvoucher=" . $this->getVoucher($cart);
            }
        }

        $dataFields[] = "data-currency=" . $this->getCurrency();

        $this->context->smarty->assign('uptain_data_fields', implode(' ', $dataFields));
        $this->context->smarty->assign('uptain_token', Configuration::get('UPTAINCONNECT_TOKEN'));

        $this->html = $this->context->smarty->fetch(
            $this->local_path . 'views/templates/hook/uptain_snippet.tpl'
        );

        return $this->html;
    }
}
