<?php

namespace Upta\Connect\Core;

use OxidEsales\Eshop\Application\Model\Address;
use OxidEsales\Eshop\Application\Model\Article;
use OxidEsales\Eshop\Application\Model\Category;
use OxidEsales\Eshop\Application\Model\CategoryList;
use OxidEsales\Eshop\Application\Model\Voucher;
use OxidEsales\Eshop\Core\DatabaseProvider;
use OxidEsales\Eshop\Core\Registry;

/**
 * Class ViewConfig
 *
 * @mixin \OxidEsales\Eshop\Core\ViewConfig
 */
class ViewConfig extends ViewConfig_parent
{
    public function uptaUseConsentManager(): bool
    {
        return Registry::getConfig()->getConfigParam('blFcUptainUseConsentManager');
    }

    public function uptaGetUserTurnover()
    {
        $user = $this->uptaGetUCUser();

        if (!Registry::getConfig()->getConfigParam('blFcTurnover')) {
            return '';
        }

        $turnover = '';
        if ($user && $user->getId() && $user->oxuser__oxregister->value > 1) {
            $oDb = DatabaseProvider::getDb();
            $sQ = 'select sum(oxtotalordersum) from oxorder where oxuserid = '
                . $oDb->quote($user->getId())
                . ' AND oxorderdate >= '
                . $oDb->quote($user->oxuser__oxregister->value)
                . ' AND oxpaid >= '
                . $oDb->quote($user->oxuser__oxregister->value)
                . ' and oxshopid = ' . $oDb->quote($this->getConfig()->getShopId());

            $result = (float)$oDb->getOne($sQ);
            if (is_numeric($result)) {
                $turnover = $this->uptaNumberFormat($result);
            }
        }

        return $turnover;
    }

    private function uptaPersonalInformationAllowed()
    {
        $user = $this->uptaGetUCUser();

        $bMailBestand = Registry::getConfig()->getConfigParam('blFcMailBestand');
        $bMailNewsletter = Registry::getConfig()->getConfigParam('blFcMailNewsletter');
        $optinStatus = $user->getNewsSubscription()->getOptInStatus();
        //$orderCount = $user->getOrderCount();
        $revenue = $this->uptaGetUserTurnover();

        return ($bMailBestand && $revenue && floatval($revenue) > 0) || ($bMailNewsletter && $optinStatus == 1);
    }

    public function uptaGetPageType()
    {
        /*
         * Must be one of the following strings: "home", "checkout", "cart", "product", "category", "success", "other", "search"
         */
        $this->getActiveClassName();

        switch ($this->getActiveClassName()) {
            case 'start':
                $sPageType = 'home';
                break;
            case 'user':
            case 'payment':
            case 'order':
                $sPageType = 'checkout';
                break;
            case 'basket':
                $sPageType = 'cart';
                break;
            case 'details':
                $sPageType = 'product';
                break;
            case 'alist':
            case 'manufacturerlist':
                $sPageType = 'category';
                break;
            case 'thankyou':
                $sPageType = 'success';
                break;
            case 'search':
                $sPageType = 'search';
                break;
            default:
                $sPageType = 'other';
        }

        return $sPageType;
    }

    /**
     * Loads uptain token from module config.
     *
     * @return string sFcToken
     */
    public function uptaGetToken()
    {
        return Registry::getConfig()->getConfigParam('sFcToken');
    }

    /**
     * Gender of the shop’s customer.
     * Use "f" for females and "m" for males.
     * Leave empty ('') if diverse or unknown
     *
     * @return string $sGender
     */
    public function uptaGetGender()
    {
        $user = $this->uptaGetUCUser();

        if (!$user) {
            return '';
        }

        $sGender = $user->oxuser__oxsal->value;

        if (!$sGender) {
            return '';
        }

        if ($this->uptaPersonalInformationAllowed()) {
            switch ($sGender) {
                case 'MR':
                    return 'm';
                    break;
                case 'MRS':
                    return 'f';
                    break;
                default:
                    return '';
            }
        } else {
            return '';
        }
    }

    public function uptaGetFirstname()
    {
        $user = $this->uptaGetUCUser();

        if (!$user) {
            return '';
        }

        $value = $user->oxuser__oxfname->value;

        if (!$value) {
            return '';
        }

        if ($this->uptaPersonalInformationAllowed()) {
            return $value;
        } else {
            return '';
        }
    }

    public function uptaGetLastname()
    {
        $user = $this->uptaGetUCUser();

        if (!$user) {
            return '';
        }

        $value = $user->oxuser__oxlname->value;

        if (!$value) {
            return '';
        }

        if ($this->uptaPersonalInformationAllowed()) {
            return $value;
        } else {
            return '';
        }
    }

    private function uptaFloatFromFormatted($str)
    {
        if (!$str) {
            return 0;
        }
        $only_digits = implode('', array_filter(str_split($str), function ($ch, $k) {
            return ctype_digit($ch);
        }, ARRAY_FILTER_USE_BOTH));
        $decimals = substr($only_digits, -2);
        $predecimals = substr($only_digits, 0, -2);
        return floatval("$predecimals.$decimals");
    }

    /**
     * Gets the current basket value
     *
     * @param $view
     *
     * @return string $scv
     */
    public function uptaGetScv($view)
    {
        $order = $this->uptaGetUCOrder($view);

        $basket = null;

        if ($order) {
            $net = $order->getOrderNetSum();
            $net -= $order->getOrderDeliveryPrice()->getNettoPrice();
            $net -= $order->getOrderWrappingPrice()->getNettoPrice();
            $net -= $order->getOrderPaymentPrice()->getNettoPrice();
            $net -= $this->uptaFloatFromFormatted($order->getFormattedTotalVouchers());

            return $this->uptaNumberFormat($net);
        } else {
            $basket = $this->uptaGetUCBasket();
        }

        if (!$basket) {
            return '';
        }

        if (count($basket->getBasketArticles()) == 0) {
            $scv = "0.00";
        } else {
            $scv = $this->uptaNumberFormat($basket->getNettoSum() - $basket->getTotalDiscountSum());
        }
        return $scv;
    }

    public function uptaGetComparison()
    {
        $items = \OxidEsales\Eshop\Core\Registry::getSession()->getVariable('aFiltcompproducts');
        $list = [];
        if (is_array($items) && count($items)) {
            $list = oxNew(\OxidEsales\Eshop\Application\Model\ArticleList::class);
            $list->loadIds(array_keys($items));
        }

        $articlesForJson = [];
        foreach ($list as $article) {
            $variants = $this->uptaGetVariants($article);
            $articlesForJson[$article->oxarticles__oxartnum->value] = ['amount' => 1, 'name' => $article->oxarticles__oxtitle->value, 'variants' => $variants];
        }

        return json_encode($articlesForJson, JSON_HEX_APOS);
    }

    public function uptaGetWishlist()
    {
        $user = $this->uptaGetUCUser();
        $list = [];

        if ($user) {
            $wishlistBasket = $user->getBasket('wishlist');
            $list = $wishlistBasket->getArticles();
        }

        $articlesForJson = [];
        foreach ($list as $article) {
            $variants = $this->uptaGetVariants($article);
            $articlesForJson[$article->oxarticles__oxartnum->value] = ['amount' => 1, 'name' => $article->oxarticles__oxtitle->value, 'variants' => $variants];
        }

        return json_encode($articlesForJson, JSON_HEX_APOS);
    }

    public function uptaGetBasketProducts()
    {
        $basket = $this->uptaGetUCBasket();

        if (!$basket) {
            return [];
        }

        $basketContent = $basket->getContents();
        if (count($basketContent) == 0) {
            return [];
        }

        $basketArticlesForJson = [];
        foreach ($basketContent as $basketItem) {

            $basketArticle = $basketItem->getArticle(true);

            $variants = $this->uptaGetVariants($basketArticle);

            /** @var Article $basketArticle */
            $basketArticlesForJson[$basketArticle->oxarticles__oxartnum->value] = ['amount' => $basketItem->getAmount(), 'name' => $basketItem->getTitle(), 'variants' => $variants];
        }

        return json_encode($basketArticlesForJson, JSON_HEX_APOS);
    }

    public function uptaGetUsedvoucher($view)
    {

        $basket = $this->uptaGetUCBasket();
        $order = $this->uptaGetUCOrder($view);

        if ($order) {
            $list = $order->getVoucherNrList();
            if (!$list) {
                return '';
            }
            return implode(',', $list);
        }

        if (!$basket) {
            return '';
        }

        $func = function ($value) {
            return $value->sVoucherNr;
        };

        $vouchers = array_map($func, $basket->getVouchers());

        if (count($vouchers) == 0) {
            return '';
        }

        return join(",", $vouchers);

    }

    public function uptaGetSuccess($view)
    {
        if ($this->uptaGetUCOrder($view)) {
            return "1";
        }
        return '';
    }

    public function uptaGetOrdernumber($view)
    {
        $order = $this->uptaGetUCOrder($view);

        if (!$order) {
            return '';
        }

        return $order->oxorder__oxordernr->value;

    }

    public function uptaGetCurrency($view)
    {
        $currencyName = $this->uptaGetUCBasket()->getBasketCurrency()->name;

        if (!$currencyName) {
            return '';
        }

        return $currencyName;
    }

    public function uptaGetReturnUrl($arg)
    {
        return urlencode($arg->getBasketLink());
    }

    public function uptaGetTaxAmount()
    {
        $fReturnVat = 0.00;

        $basket = $this->uptaGetUCBasket();
        if (!$basket) {
            return $fReturnVat;
        }

        foreach ($basket->getProductVats(false) as $fVat) {
            $fReturnVat += $fVat;
        }

        return $this->uptaNumberFormat($fReturnVat);
    }

    public function uptaGetVoucherType()
    {
        $basket = $this->uptaGetUCBasket();
        if (!$basket) {
            return '';
        }

        $aVouchers = $basket->getVouchers();
        if (count($aVouchers)) {
            foreach (array_keys($aVouchers) as $sVoucherId) {
                $oVoucher = oxNew(Voucher::class);
                $oVoucher->load($sVoucherId);

                $oSeries = $oVoucher->getSerie();
                switch ($oSeries->oxvoucherseries__oxdiscounttype->value) {
                    case 'percent':
                        return 'percentage';
                    case 'absolute':
                        return 'monetary';
                    default:
                        return 'other';
                }

            }
        }

        return '';
    }

    public function uptaGetVoucherAmount()
    {

        $basket = $this->uptaGetUCBasket();

        if (!$basket) {
            return '';
        }

        $voucherDiscount = $basket->getVoucherDiscount();
        if (!$voucherDiscount) {
            return '';
        }

        return $voucherDiscount->getBruttoPrice();
    }

    public function uptaGetPostalCode()
    {
        $user = $this->uptaGetUCUser();
        if (!$user) {
            return '';
        }

        $delZip = '';
        $sAddressId = $this->uptaGetUCSession()->getVariable('deladrid');
        $oDeliveryAddress = oxNew(Address::class);
        if ($oDeliveryAddress->load($sAddressId)) {
            $delZip = $oDeliveryAddress->oxaddress__oxzip->value;
        }
        return sprintf('delivery:%s;accounting:%s', $delZip ?: $user->oxuser__oxzip->value, $user->oxuser__oxzip->value);
    }

    public function uptaGetCategoryPath($view)
    {
        if ($view instanceof \OxidEsales\Eshop\Application\Controller\ArticleListController) {

            $categoryPath = '';
            if (($categoryTree = $view->getCategoryTree()) && ($categoryPaths = $categoryTree->getPath())) {

                $i = 1;
                foreach ($categoryPaths as $category) {
                    /** @var Category $category */
                    if (count($categoryPaths) > $i) {
                        $categoryPath .= $category->oxcategories__oxtitle->value;
                    }
                    if (count($categoryPaths) - 1 > $i) {
                        $categoryPath .= ';';
                    }
                    $i++;

                }
            }

            return $categoryPath;
        } elseif ($view instanceof \OxidEsales\Eshop\Application\Controller\ArticleDetailsController) {

            /** @var Article $article */
            $article = $view->getProduct();
            $categoryIds = $article->getCategoryIds();
            $categoryPath = '';

            foreach ($categoryIds as $categoryid) {


                /** @var CategoryList $categoryTree */
                $categoryTree = oxNew(CategoryList::class);
                $categoryTree->buildTree($categoryid);


                if ($categoryPaths = $categoryTree->getPath()) {

                    $i = 1;
                    foreach ($categoryPaths as $category) {
                        /** @var Category $category */
                        if (count($categoryPaths) > $i) {
                            if ($i == 1 && $categoryPath != '') {
                                $categoryPath .= ';';
                            }
                            $categoryPath .= $category->oxcategories__oxtitle->value;
                        }
                        if (count($categoryPaths) - 1 > $i) {
                            $categoryPath .= '/';
                        }
                        $i++;

                    }
                }
            }

            return $categoryPath;
        }

        return '';
    }

    public function uptaGetCategoryProducts($view)
    {
        if ($view instanceof \OxidEsales\Eshop\Application\Controller\ArticleListController) {

            $categoryArticlesForJson = [];
            $i = 1;
            foreach ($view->getArticleList() as $key => $article) {
                /** @var Article $article */
                $categoryArticlesForJson[$article->oxarticles__oxartnum->value] = ['name' => $article->oxarticles__oxtitle->value, 'position' => $i];
                $i++;
            }

            return json_encode($categoryArticlesForJson, JSON_HEX_APOS);
        }

        return '';
    }

    public function uptaGetCategorySorting($view)
    {
        if ($view instanceof \OxidEsales\Eshop\Application\Controller\ArticleListController) {

            return $view->getListOrderBy() . '-' . $view->getListOrderDirection();
        }

        return '';
    }

    public function uptaNumberFormat($number)
    {
        return number_format($number, 2, '.', '');
    }

    public function uptaGetArticleVariants($view)
    {
	    $variants = [];

        if ($view instanceof \OxidEsales\Eshop\Application\Controller\ArticleDetailsController) {

            /** @var Article $article */
            $article = $view->getProduct();

            $variants = $this->uptaGetVariants($article);
        }

        return json_encode($variants, JSON_HEX_APOS);
    }

    public function uptaGetSearchSorting($view)
    {
        if ($view instanceof \OxidEsales\Eshop\Application\Controller\SearchController) {

            return $view->getListOrderBy() . '-' . $view->getListOrderDirection();
        }

        return '';
    }

    public function uptaGetSearchProducts($view)
    {
        if ($view instanceof \OxidEsales\Eshop\Application\Controller\SearchController) {

            $categoryArticlesForJson = [];
            $i = 1;
            foreach ($view->getArticleList() as $key => $article) {
                /** @var Article $article */
                $categoryArticlesForJson[$article->oxarticles__oxartnum->value] = ['name' => $article->oxarticles__oxtitle->value, 'position' => $i];
                $i++;
            }

            return json_encode($categoryArticlesForJson, JSON_HEX_APOS);
        }

        return '';
    }

    public function uptaGetPayment()
    {

        $basket = $this->uptaGetUCBasket();

        if (!$basket) {
            return '';
        }

        return $basket->getPaymentId();

    }

    public function uptaGetShipping()
    {

        $basket = $this->uptaGetUCBasket();

        if (!$basket) {
            return '';
        }

        return $basket->getShippingId();

    }

    public function uptaGetDeliveryCosts()
    {

        $basket = $this->uptaGetUCBasket();

        if (!$basket) {
            return '';
        }

        $delivery = $basket->getCosts('oxdelivery');

        if (!$delivery) {
            return '';
        }

        $c = $delivery->getBruttoPrice();

        if ($c == 0) {
            return '';
        }

        return $c;

    }

    public function uptaGetPaymentCosts()
    {

        $basket = $this->uptaGetUCBasket();

        if (!$basket) {
            return '';
        }

        $payment = $basket->getCosts('oxpayment');

        if (!$payment) {
            return '';
        }

        $c = $payment->getBruttoPrice();

        if ($c == 0) {
            return '';
        }

        return $c;

    }

    public function uptaGetEmail()
    {

        $user = $this->uptaGetUCUser();

        if (!$user) {
            return '';
        }

        $mail = $user->oxuser__oxusername->value;
        if ($this->uptaPersonalInformationAllowed()) {
            return $mail;
        }
        return '';
    }

    public function uptaGetUniqueId()
    {
        $user = $this->uptaGetUCUser();

        if (!$user || !Registry::getConfig()->getConfigParam('blFcUniqueId')) {
            return '';
        }

        $id = md5($user->getId());

        return $id;
    }

    public function uptaGetCustomerGroup()
    {
        $user = $this->uptaGetUCUser();
        if (!$user) {
            return '';
        }

        $groups = $user->getUserGroups();
        $groupNames = [];
        if ($groups) {
            foreach ($groups as $g) {
                $groupNames[] = $g->oxgroups__oxtitle->value;
            }
        }

        return implode(',', $groupNames);
    }

    public function uptaGetPlugin($view)
    {
        return "oxid" . $view->getShopVersion() . ":" . Registry::getConfig()->getConfigParam('aModuleVersions')["uptaconnect"];
    }


    private function uptaGetUCOrder($view)
    {
        $shopVersion = $view->getShopVersion();

        if (explode('.', $shopVersion)[0] >= 6) {
            if ($view instanceof \OxidEsales\Eshop\Application\Controller\ThankYouController) {
                return $view->getOrder();
            }
            return null;
        } else {
            if ($view instanceof Thankyou) {
                return $view->getOrder();
            }
            return null;
        }
    }

    private function uptaGetUCBasket()
    {
        return $this->uptaGetUCSession()->getBasket();
    }

    private function uptaGetUCUser()
    {
        return $this->uptaGetUCSession()->getUser();
    }

    private function uptaGetUCSession()
    {
        return Registry::getSession();
    }

    private function uptaGetVariants($article)
    {
	    // Check config
	    if (!Registry::getConfig()->getConfigParam('blFcUptaShowVariants')){
		    return '';
	    }

	    // Load variants
        $variants = [];
        $parentId = $article->oxarticles__oxparentid->value;

        if ($parentId) {
            $articleParent = oxNew(Article::class);
            $articleParent->load($parentId);
        } else {
            $articleParent = $article;
        }

        //gets all variants names
        $variantsNames = explode(' | ', $articleParent->oxarticles__oxvarname->value);

        if (count($variantsNames) > 0) {
            //gets all variants options
            $variantsOptions = [];
            foreach ($articleParent->getSimpleVariants() as $articleVariant) {
                $variantsOptions[] = explode(' | ', $articleVariant->oxarticles__oxvarselect->value);
            }

            //merge names and options to one array
            foreach ($variantsOptions as $variantOptions) {
                foreach ($variantOptions as $key => $variantOption) {
                	if (!is_array($variants[$variantsNames[$key]])){
		                $variants[$variantsNames[$key]] = [];
	                }
                    if (!in_array($variantOption, $variants[$variantsNames[$key]])) {
                        $variants[$variantsNames[$key]][] = $variantOption;
                    }
                }
            }
        }

        return $variants;
    }
}

